<?php
// Konfiguracja
 $rootDir = 'X'; // Główny folder ze zdjęciami
 $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp']; // Dozwolone formaty zdjęć

// Funkcja do wykrywania urządzeń mobilnych na podstawie User Agent
function isMobile() {
    return preg_match("/(android|avantgo|blackberry|bolt|boost|cricket|docomo|fone|hiptop|mini|mobi|palm|phone|pie|tablet|up\.browser|up\.link|webos|wos)/i", $_SERVER["HTTP_USER_AGENT"]);
}

// Ustawienie liczby elementów na stronę w zależności od urządzenia
 $itemsPerPage = isMobile() ? 12 : 24;

// Funkcja do rekurencyjnego skanowania folderów
function scanDirectory($dir, $recursive = false) {
    global $rootDir, $allowedExtensions;

    $result = [
        'folders' => [],
        'images' => []
    ];

    if (!is_dir($dir)) {
        return $result;
    }

    $items = scandir($dir);

    foreach ($items as $item) {
        if ($item == '.' || $item == '..') continue;

        $path = $dir . '/' . $item;

        if (is_dir($path)) {
            $folderData = [
                'name' => $item,
                'path' => str_replace($rootDir . '/', '', $path)
            ];

            if ($recursive) {
                $folderData['subfolders'] = scanDirectory($path, true)['folders'];
            }

            $result['folders'][] = $folderData;
        } else {
            $ext = strtolower(pathinfo($item, PATHINFO_EXTENSION));
            if (in_array($ext, $allowedExtensions)) {
                $result['images'][] = $item;
            }
        }
    }

    // Sortowanie
    usort($result['folders'], function($a, $b) {
        return strnatcasecmp($a['name'], $b['name']);
    });

    usort($result['images'], function($a, $b) use ($dir) {
        return filemtime($dir . '/' . $a) - filemtime($dir . '/' . $b);
    });

    return $result;
}

// Pobierz zawartość folderu jeśli jest żądanie AJAX
if (isset($_GET['action']) && $_GET['action'] == 'get_folder') {
    $folder = isset($_GET['folder']) ? $_GET['folder'] : '';
    $recursive = isset($_GET['recursive']) && $_GET['recursive'] == 'true';
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $dir = $rootDir . ($folder ? '/' . $folder : '');

    $data = scanDirectory($dir, $recursive);
    
    // Paginacja zdjęć
    if (!$recursive) {
        $totalImages = count($data['images']);
        $totalPages = ceil($totalImages / $itemsPerPage);
        $offset = ($page - 1) * $itemsPerPage;
        
        $data['pagination'] = [
            'currentPage' => $page,
            'totalPages' => $totalPages,
            'totalImages' => $totalImages,
            'itemsPerPage' => $itemsPerPage // Przekazanie informacji do JS
        ];
        
        // Zapisz wszystkie zdjęcia do odpowiedzi dla lightboxa
        $data['allImages'] = $data['images'];
        
        $data['images'] = array_slice($data['images'], $offset, $itemsPerPage);
    }

    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

// Sprawdź czy jest bezpośrednie żądanie konkretnego folderu
 $initialFolder = isset($_GET['folder']) ? $_GET['folder'] : '';
?>
<!DOCTYPE html>
<html lang="pl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>GALERIA ZDJĘĆ - XYZ</title>
    <style>
        :root {
            --bg-color: #000;
            --text-color: #90ee90;
            --panel-color: #111;
            --border-color: #333;
            --highlight-color: #444;
            --button-color: #333;
            --gallery-item-height: 200px;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Courier New', Courier, monospace;
            background-color: var(--bg-color);
            color: var(--text-color);
            display: flex;
            flex-direction: column;
            min-height: 100vh;
            touch-action: manipulation;
        }

        header {
            padding: 15px;
            text-align: center;
            background-color: var(--panel-color);
            border-bottom: 1px solid var(--border-color);
        }

        h1 {
            font-size: 1.5rem;
            font-weight: bold;
            text-shadow: 0 0 5px var(--text-color);
        }

        .subtitle {
            font-size: 0.75rem; /* 2x mniejsza niż tytuł (1.5rem / 2) */
            color: var(--text-color);
            opacity: 0.8;
            margin-top: 5px;
        }

        .container {
            display: flex;
            flex: 1;
            flex-direction: row;
            overflow: hidden;
        }

        .explorer {
            width: 400px;
            background-color: var(--panel-color);
            padding: 10px;
            overflow-y: auto;
            border-right: 1px solid var(--border-color);
        }

        .gallery {
            flex: 1;
            padding: 15px;
            overflow-y: auto;
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            grid-auto-rows: min-content;
            gap: 5px;
            align-items: start;
        }

        .folder-item, .file-item {
            padding: 8px;
            cursor: pointer;
            border-radius: 3px;
            margin-bottom: 5px;
            transition: background-color 0.2s;
        }

        .folder-item:hover, .file-item:hover {
            background-color: var(--highlight-color);
        }

        .folder-item::before {
            content: "●";
        }

        .current-folder {
            background-color: var(--highlight-color);
            font-weight: bold;
        }

        .gallery-item {
            display: flex;
            justify-content: center;
            align-items: center;
            background-color: var(--panel-color);
            padding: 2px;
            border-radius: 3px;
            transition: transform 0.3s;
            height: var(--gallery-item-height);
            overflow: hidden;
            position: relative;
        }

        .gallery-item:hover {
            transform: scale(1.02);
        }

        .gallery-item img {
            max-width: 100%;
            max-height: 100%;
            width: auto;
            height: auto;
            object-fit: contain;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            opacity: 0;
            transition: opacity 0.3s;
        }

        .gallery-item img.loaded {
            opacity: 1;
        }

        .gallery-item .placeholder {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: #222;
            display: flex;
            justify-content: center;
            align-items: center;
            color: #555;
            font-size: 14px;
        }

        .gallery-item .spinner-container {
            position: relative;
            width: 50px;
            height: 50px;
        }

        .gallery-item .spinner {
            border: 3px solid rgba(255, 255, 255, 0.1);
            border-radius: 50%;
            border-top: 3px solid var(--text-color);
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
        }

        .gallery-item .progress-percentage {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            color: var(--text-color);
            font-size: 12px;
            font-weight: bold;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .lightbox {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.95);
            z-index: 1000;
            justify-content: center;
            align-items: center;
            touch-action: pan-y;
        }

        .lightbox-content {
            max-width: 90%;
            max-height: 90%;
            position: relative;
            text-align: center;
        }

        .lightbox-img {
            max-width: 90vw;
            max-height: 90vh;
            object-fit: contain;
            opacity: 0;
            transition: opacity 0.3s;
        }

        .lightbox-img.loaded {
            opacity: 1;
        }

        .lightbox-spinner {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            border: 5px solid rgba(255, 255, 255, 0.1);
            border-radius: 50%;
            border-top: 5px solid white;
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
        }

        .close-lightbox {
            position: absolute;
            top: 20px;
            right: 20px;
            font-size: 30px;
            color: white;
            cursor: pointer;
            background-color: rgba(0, 0, 0, 0.5);
            width: 40px;
            height: 40px;
            text-align: center;
            line-height: 40px;
            border-radius: 50%;
            z-index: 1002;
        }

        .nav-lightbox {
            position: fixed;
            top: 50%;
            transform: translateY(-50%);
            font-size: 30px;
            color: white;
            cursor: pointer;
            background-color: rgba(0, 0, 0, 0.5);
            width: 50px;
            height: 50px;
            text-align: center;
            line-height: 50px;
            border-radius: 50%;
            user-select: none;
            z-index: 1002;
            -webkit-tap-highlight-color: transparent;
        }

        .prev {
            left: 10px;
        }

        .next {
            right: 10px;
        }

        .footer {
            padding: 10px;
            background-color: var(--panel-color);
            text-align: center;
            border-top: 1px solid var(--border-color);
        }

        .gallery-url-container {
            margin-top: 10px;
            display: flex;
            flex-direction: column;
            align-items: center;
        }

        .gallery-url {
            padding: 8px;
            background-color: var(--button-color);
            word-break: break-all;
            border-radius: 3px;
            width: 90%;
            max-width: 600px;
        }

        .copy-btn {
            margin-top: 10px;
            padding: 8px 15px;
            background-color: var(--button-color);
            color: var(--text-color);
            border: none;
            border-radius: 3px;
            cursor: pointer;
            transition: background-color 0.2s;
        }

        .copy-btn:hover {
            background-color: var(--highlight-color);
        }

        .copy-success {
            margin-top: 5px;
            color: #4CAF50;
            display: none;
        }

        .empty-message {
            width: 100%;
            text-align: center;
            padding: 20px;
            color: #777;
            grid-column: 1 / -1;
        }

        .subfolders {
            margin-left: 15px;
            border-left: 1px solid var(--border-color);
            padding-left: 10px;
            display: none;
        }

        .folder-toggle {
            margin-right: 5px;
            cursor: pointer;
            display: inline-block;
            width: 15px;
            text-align: center;
        }

        .folder-expanded .subfolders {
            display: block;
        }

        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            flex-wrap: wrap;
            margin-top: 20px;
            gap: 5px;
            grid-column: 1 / -1;
            max-width: 100%;
            overflow-x: auto;
            padding: 5px;
        }

        .pagination button {
            padding: 8px 12px;
            background-color: var(--button-color);
            color: var(--text-color);
            border: none;
            border-radius: 3px;
            cursor: pointer;
            transition: background-color 0.2s;
            min-width: 40px;
            font-size: 14px;
            white-space: nowrap;
        }

        .pagination button:hover {
            background-color: var(--highlight-color);
        }

        .pagination button.active {
            background-color: var(--highlight-color);
            font-weight: bold;
        }

        .pagination button:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }

        .pagination .nav-btn {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 8px 10px;
        }

        .pagination .nav-btn .btn-text {
            display: inline;
        }

        .pagination .nav-btn .btn-icon {
            display: none;
            font-size: 16px;
        }

        .pagination .dots {
            padding: 0 5px;
            color: var(--text-color);
            opacity: 0.7;
        }

        .load-more {
            padding: 10px;
            background-color: var(--button-color);
            color: var(--text-color);
            border: none;
            border-radius: 3px;
            cursor: pointer;
            margin: 20px auto;
            display: block;
            transition: background-color 0.2s;
            grid-column: 1 / -1;
        }

        .load-more:hover {
            background-color: var(--highlight-color);
        }

        /* Responsywność */
        @media (max-width: 1024px) {
            .gallery {
                grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            }

            :root {
                --gallery-item-height: 180px;
            }
        }

        @media (max-width: 768px) {
            .container {
                flex-direction: column;
            }

            .explorer {
                width: 100%;
                max-height: 200px;
                border-right: none;
                border-bottom: 1px solid var(--border-color);
            }

            .gallery {
                grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
                gap: 3px;
            }

            :root {
                --gallery-item-height: 160px;
            }

            .nav-lightbox {
                width: 40px;
                height: 40px;
                line-height: 40px;
                font-size: 25px;
            }

            .prev {
                left: 5px;
            }

            .next {
                right: 5px;
            }

            .pagination {
                gap: 3px;
                padding: 3px;
            }

            .pagination button {
                padding: 6px 8px;
                font-size: 12px;
                min-width: 32px;
            }

            .pagination .nav-btn {
                padding: 6px 8px;
            }

            .pagination .nav-btn .btn-text {
                display: none;
            }

            .pagination .nav-btn .btn-icon {
                display: inline;
            }
        }

        @media (max-width: 480px) {
            .gallery {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            }

            h1 {
                font-size: 1.2rem;
            }

            .subtitle {
                font-size: 0.6rem; /* 1.2rem / 2 */
            }

            :root {
                --gallery-item-height: 140px;
            }

            .nav-lightbox {
                width: 35px;
                height: 35px;
                line-height: 35px;
                font-size: 20px;
            }

            .close-lightbox {
                width: 35px;
                height: 35px;
                line-height: 35px;
                font-size: 25px;
            }

            .pagination {
                gap: 2px;
                padding: 2px;
            }

            .pagination button {
                padding: 5px 6px;
                font-size: 11px;
                min-width: 28px;
            }

            .pagination .nav-btn {
                padding: 5px 6px;
            }

            .pagination .dots {
                padding: 0 3px;
                font-size: 11px;
            }
        }

        @media (max-width: 360px) {
            .pagination {
                justify-content: center;
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }

            .pagination button {
                padding: 4px 5px;
                font-size: 10px;
                min-width: 26px;
            }

            .pagination .nav-btn {
                padding: 4px 5px;
            }
        }

        /* Styl dla touch devices */
        @media (hover: none) and (pointer: coarse) {
            .nav-lightbox {
                background-color: rgba(0, 0, 0, 0.7);
                width: 60px;
                height: 60px;
                line-height: 60px;
                font-size: 30px;
            }

            .prev {
                left: 15px;
            }

            .next {
                right: 15px;
            }
        }
    </style>
</head>
<body oncontextmenu="return false;">
    <header>
        <h1>GALERIA ZDJĘĆ - XYZ</h1>
        <p class="subtitle">Jeśli widzisz kółko postępu ładowania galerii to poczekaj proszę...</p>
    </header>

    <div class="container">
        <div class="explorer" id="explorer">
        </div>

        <div class="gallery" id="gallery">
            <div class="empty-message">Wybierz folder z lewej strony, aby zobaczyć zdjęcia</div>
        </div>
    </div>

    <div class="footer">
        <div class="gallery-url-container">
            <div class="gallery-url" id="galleryUrl"></div>
            <button class="copy-btn" id="copyBtn">Kopiuj adres galerii</button>
            <div class="copy-success" id="copySuccess">Adres skopiowany do schowka!</div>
        </div>
    </div>

    <div class="lightbox" id="lightbox">
        <span class="close-lightbox">&times;</span>
        <span class="nav-lightbox prev" id="prevPhoto">&lt;</span>
        <div class="lightbox-content">
            <div class="lightbox-spinner" id="lightboxSpinner"></div>
            <img class="lightbox-img" id="lightboxImage" src="" alt="">
        </div>
        <span class="nav-lightbox next" id="nextPhoto">&gt;</span>
    </div>

    <script>
        // Przekazanie wartości itemsPerPage z PHP do JavaScript
        const itemsPerPage = <?php echo $itemsPerPage; ?>;
        
        // Blokada prawego przycisku myszy
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });

        document.addEventListener('DOMContentLoaded', function() {
            const explorer = document.getElementById('explorer');
            const gallery = document.getElementById('gallery');
            const lightbox = document.getElementById('lightbox');
            const lightboxImage = document.getElementById('lightboxImage');
            const lightboxSpinner = document.getElementById('lightboxSpinner');
            const closeLightbox = document.querySelector('.close-lightbox');
            const prevPhoto = document.getElementById('prevPhoto');
            const nextPhoto = document.getElementById('nextPhoto');
            const galleryUrl = document.getElementById('galleryUrl');
            const copyBtn = document.getElementById('copyBtn');
            const copySuccess = document.getElementById('copySuccess');

            let currentFolder = '';
            let currentImages = [];
            let currentImageIndex = 0;
            let touchStartX = 0;
            let touchEndX = 0;
            let currentPage = 1;
            let totalPages = 1;
            let isLoadingMore = false;
            let preloadedImages = {};
            let allImages = []; // Wszystkie zdjęcia w folderze (do nawigacji w lightboxie)

            // Obserwator Intersection Observer dla lazy loading
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        const placeholder = img.previousElementSibling;
                        
                        // Rozpocznij ładowanie obrazka z śledzeniem postępu
                        loadImageWithProgress(img, placeholder);
                        
                        observer.unobserve(img);
                    }
                });
            }, {
                rootMargin: '50px' // Zacznij ładować 50px przed widocznym obszarem
            });

            // Funkcja do ładowania obrazka ze śledzeniem postępu
            function loadImageWithProgress(img, placeholder) {
                const progressPercentage = placeholder.querySelector('.progress-percentage');
                
                // Utwórz obiekt XMLHttpRequest do śledzenia postępu
                const xhr = new XMLHttpRequest();
                xhr.open('GET', img.dataset.src, true);
                xhr.responseType = 'blob';
                
                xhr.onprogress = function(e) {
                    if (e.lengthComputable) {
                        const percentComplete = Math.round((e.loaded / e.total) * 100);
                        progressPercentage.textContent = percentComplete + '%';
                    }
                };
                
                xhr.onload = function() {
                    if (this.status === 200) {
                        const blob = this.response;
                        const url = URL.createObjectURL(blob);
                        
                        img.onload = function() {
                            img.classList.add('loaded');
                            if (placeholder) {
                                placeholder.style.display = 'none';
                            }
                            URL.revokeObjectURL(url);
                        };
                        
                        img.onerror = function() {
                            if (placeholder) {
                                placeholder.innerHTML = '<span>Błąd ładowania</span>';
                            }
                            URL.revokeObjectURL(url);
                        };
                        
                        img.src = url;
                    } else {
                        if (placeholder) {
                            placeholder.innerHTML = '<span>Błąd ładowania</span>';
                        }
                    }
                };
                
                xhr.onerror = function() {
                    if (placeholder) {
                        placeholder.innerHTML = '<span>Błąd ładowania</span>';
                    }
                };
                
                xhr.send();
            }

            // Ładowanie struktury folderów
            function loadFolderStructure(path = '', parentElement = explorer, isInitialLoad = false) {
                fetch(`?action=get_folder&folder=${encodeURIComponent(path)}&recursive=true`)
                    .then(response => response.json())
                    .then(data => {
                        if (!isInitialLoad) {
                            parentElement.innerHTML = '';
                        }

                        data.folders.forEach(folder => {
                            const folderElement = createFolderElement(folder);
                            parentElement.appendChild(folderElement);
                        });

                        // Jeśli mamy początkowy folder w URL i to jest pierwsze ładowanie
                        if (isInitialLoad && initialFolder) {
                            const folderElement = document.querySelector(`.folder-item[data-path="${initialFolder}"]`);
                            if (folderElement) {
                                const folderName = folderElement.querySelector('.folder-name').textContent;
                                loadGallery(folderName, initialFolder);
                            } else if (path === '' && data.folders.length > 0) {
                                // Jeśli folder z URL nie istnieje, załaduj pierwszy folder
                                loadGallery(data.folders[0].name, data.folders[0].path);
                            }
                        } else if (isInitialLoad && path === '' && data.folders.length > 0 && !initialFolder) {
                            // Automatycznie załaduj pierwszy folder przy pierwszym ładowaniu, jeśli nie ma folderu w URL
                            loadGallery(data.folders[0].name, data.folders[0].path);
                        }
                    })
                    .catch(error => {
                        console.error('Błąd ładowania folderów:', error);
                    });
            }

            // Tworzenie elementu folderu z możliwością rozwijania
            function createFolderElement(folder) {
                const folderElement = document.createElement('div');
                folderElement.className = 'folder-item';
                folderElement.dataset.path = folder.path;

                const folderContent = document.createElement('div');
                folderContent.className = 'folder-content';

                const toggle = document.createElement('span');
                toggle.className = 'folder-toggle';
                toggle.textContent = folder.subfolders && folder.subfolders.length > 0 ? '▶' : '';
                toggle.addEventListener('click', function(e) {
                    e.stopPropagation();
                    folderElement.classList.toggle('folder-expanded');
                    toggle.textContent = folderElement.classList.contains('folder-expanded') ? '▼' : '▶';
                });

                const nameSpan = document.createElement('span');
                nameSpan.className = 'folder-name';
                nameSpan.textContent = folder.name;

                nameSpan.addEventListener('click', function() {
                    loadGallery(folder.name, folder.path);
                });

                folderContent.appendChild(toggle);
                folderContent.appendChild(nameSpan);
                folderElement.appendChild(folderContent);

                if (folder.subfolders && folder.subfolders.length > 0) {
                    const subfoldersContainer = document.createElement('div');
                    subfoldersContainer.className = 'subfolders';

                    folder.subfolders.forEach(subfolder => {
                        const subfolderElement = createFolderElement(subfolder);
                        subfoldersContainer.appendChild(subfolderElement);
                    });

                    folderElement.appendChild(subfoldersContainer);
                }

                return folderElement;
            }

            // Ładowanie galerii zdjęć
            function loadGallery(folderName, folderPath, page = 1, append = false) {
                currentFolder = folderPath;
                currentPage = page;

                fetch(`?action=get_folder&folder=${encodeURIComponent(folderPath)}&page=${page}`)
                    .then(response => response.json())
                    .then(data => {
                        if (!append) {
                            gallery.innerHTML = '';
                            currentImages = [];
                            
                            // Zapisz wszystkie zdjęcia z folderu dla lightboxa
                            if (data.allImages) {
                                allImages = data.allImages;
                            }
                        }
                        
                        // Zapisz wszystkie zdjęcia z odpowiedzi
                        const newImages = data.images;
                        currentImages = append ? [...currentImages, ...newImages] : newImages;
                        
                        // Jeśli to pierwsza strona, zapisz informacje o paginacji
                        if (!append && data.pagination) {
                            totalPages = data.pagination.totalPages;
                        }
                        
                        // Wyświetl zdjęcia
                        displayImages(newImages, folderPath, append);
                        
                        // Jeśli to nie jest pierwsza strona, dodaj przycisk "Załaduj więcej"
                        if (append && currentPage < totalPages) {
                            addLoadMoreButton();
                        } else if (!append && totalPages > 1) {
                            addPagination();
                        }

                        // Aktualizacja URL
                        if (!append) {
                            updateGalleryUrl(folderPath);
                            
                            // Zaznacz aktywny folder
                            document.querySelectorAll('.folder-item').forEach(item => {
                                item.classList.remove('current-folder');
                            });

                            const activeFolder = document.querySelector(`.folder-item[data-path="${folderPath}"]`);
                            if (activeFolder) {
                                activeFolder.classList.add('current-folder');

                                // Rozwiń wszystkie foldery nadrzędne
                                let parent = activeFolder.parentElement.closest('.folder-item');
                                while (parent) {
                                    parent.classList.add('folder-expanded');
                                    const toggle = parent.querySelector('.folder-toggle');
                                    if (toggle) toggle.textContent = '▼';
                                    parent = parent.parentElement.closest('.folder-item');
                                }
                            }
                        }
                    })
                    .catch(error => {
                        console.error('Błąd ładowania zdjęć:', error);
                        if (!append) {
                            gallery.innerHTML = '<div class="empty-message">Wystąpił błąd podczas ładowania zdjęć</div>';
                        }
                    });
            }

            // Wyświetlanie zdjęć
            function displayImages(images, folderPath, append = false) {
                if (images.length === 0 && !append) {
                    gallery.innerHTML = '<div class="empty-message">Brak zdjęć w wybranym folderze</div>';
                    return;
                }

                images.forEach((image, index) => {
                    const imgPath = `<?php echo $rootDir; ?>/${folderPath}/${image}`;
                    
                    // Znajdź indeks tego zdjęcia w allImages
                    const allImagesIndex = allImages.indexOf(image);

                    const galleryItem = document.createElement('div');
                    galleryItem.className = 'gallery-item';

                    // Dodaj placeholder z spinnerem i procentami
                    const placeholder = document.createElement('div');
                    placeholder.className = 'placeholder';
                    
                    const spinnerContainer = document.createElement('div');
                    spinnerContainer.className = 'spinner-container';
                    
                    const spinner = document.createElement('div');
                    spinner.className = 'spinner';
                    
                    const progressPercentage = document.createElement('div');
                    progressPercentage.className = 'progress-percentage';
                    progressPercentage.textContent = '0%';
                    
                    spinnerContainer.appendChild(spinner);
                    spinnerContainer.appendChild(progressPercentage);
                    placeholder.appendChild(spinnerContainer);
                    
                    galleryItem.appendChild(placeholder);

                    const img = document.createElement('img');
                    img.dataset.src = imgPath; // Używamy data-src zamiast src
                    img.alt = image;
                    
                    // Dodaj atrybuty do identyfikacji zdjęcia w lightboxie
                    img.dataset.index = allImagesIndex;
                    
                    img.addEventListener('click', function() {
                        openLightbox(parseInt(this.dataset.index));
                    });

                    galleryItem.appendChild(img);
                    gallery.appendChild(galleryItem);

                    // Obserwuj obrazek za pomocą Intersection Observer
                    imageObserver.observe(img);
                });
            }

            // Dodawanie przycisku "Załaduj więcej"
            function addLoadMoreButton() {
                const loadMoreBtn = document.createElement('button');
                loadMoreBtn.className = 'load-more';
                loadMoreBtn.textContent = 'Załaduj więcej zdjęć';
                loadMoreBtn.addEventListener('click', function() {
                    if (currentPage < totalPages && !isLoadingMore) {
                        isLoadingMore = true;
                        this.textContent = 'Ładowanie...';
                        
                        loadGallery('', currentFolder, currentPage + 1, true).then(() => {
                            isLoadingMore = false;
                            this.remove();
                            
                            if (currentPage < totalPages) {
                                addLoadMoreButton();
                            }
                        });
                    }
                });
                
                gallery.appendChild(loadMoreBtn);
            }

            // Dodawanie paginacji
            function addPagination() {
                const pagination = document.createElement('div');
                pagination.className = 'pagination';
                
                // Przycisk "Poprzednia"
                const prevBtn = document.createElement('button');
                prevBtn.className = 'nav-btn';
                prevBtn.disabled = currentPage === 1;
                prevBtn.innerHTML = '<span class="btn-text">Poprzednia</span><span class="btn-icon">‹</span>';
                prevBtn.addEventListener('click', function() {
                    if (currentPage > 1) {
                        loadGallery('', currentFolder, currentPage - 1);
                    }
                });
                pagination.appendChild(prevBtn);
                
                // Numery stron - dostosuj liczbę widocznych stron do szerokości ekranu
                const isMobile = window.innerWidth <= 768;
                const isSmallMobile = window.innerWidth <= 480;
                const maxVisiblePages = isSmallMobile ? 3 : (isMobile ? 4 : 5);
                
                let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
                let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
                
                if (endPage - startPage < maxVisiblePages - 1) {
                    startPage = Math.max(1, endPage - maxVisiblePages + 1);
                }
                
                if (startPage > 1) {
                    const firstPageBtn = document.createElement('button');
                    firstPageBtn.textContent = '1';
                    firstPageBtn.addEventListener('click', function() {
                        loadGallery('', currentFolder, 1);
                    });
                    pagination.appendChild(firstPageBtn);
                    
                    if (startPage > 2) {
                        const dots = document.createElement('span');
                        dots.className = 'dots';
                        dots.textContent = '...';
                        pagination.appendChild(dots);
                    }
                }
                
                for (let i = startPage; i <= endPage; i++) {
                    const pageBtn = document.createElement('button');
                    pageBtn.textContent = i;
                    pageBtn.classList.toggle('active', i === currentPage);
                    pageBtn.addEventListener('click', function() {
                        loadGallery('', currentFolder, i);
                    });
                    pagination.appendChild(pageBtn);
                }
                
                if (endPage < totalPages) {
                    if (endPage < totalPages - 1) {
                        const dots = document.createElement('span');
                        dots.className = 'dots';
                        dots.textContent = '...';
                        pagination.appendChild(dots);
                    }
                    
                    const lastPageBtn = document.createElement('button');
                    lastPageBtn.textContent = totalPages;
                    lastPageBtn.addEventListener('click', function() {
                        loadGallery('', currentFolder, totalPages);
                    });
                    pagination.appendChild(lastPageBtn);
                }
                
                // Przycisk "Następna"
                const nextBtn = document.createElement('button');
                nextBtn.className = 'nav-btn';
                nextBtn.disabled = currentPage === totalPages;
                nextBtn.innerHTML = '<span class="btn-text">Następna</span><span class="btn-icon">›</span>';
                nextBtn.addEventListener('click', function() {
                    if (currentPage < totalPages) {
                        loadGallery('', currentFolder, currentPage + 1);
                    }
                });
                pagination.appendChild(nextBtn);
                
                gallery.appendChild(pagination);
            }

            // Preload obrazków dla lightboxa
            function preloadImage(index) {
                if (index < 0 || index >= allImages.length) return;
                
                const imageKey = `${currentFolder}_${index}`;
                if (!preloadedImages[imageKey]) {
                    const img = new Image();
                    img.src = `<?php echo $rootDir; ?>/${currentFolder}/${allImages[index]}`;
                    preloadedImages[imageKey] = img;
                }
            }

            // Otwieranie lightbox
            function openLightbox(index) {
                currentImageIndex = index;
                lightbox.style.display = 'flex';
                document.body.style.overflow = 'hidden';
                
                // Pokaż spinner
                lightboxSpinner.style.display = 'block';
                lightboxImage.classList.remove('loaded');
                
                // Ustaw źródło obrazka
                const imgPath = `<?php echo $rootDir; ?>/${currentFolder}/${allImages[index]}`;
                
                // Sprawdź, czy obrazek jest już załadowany
                const imageKey = `${currentFolder}_${index}`;
                if (preloadedImages[imageKey]) {
                    lightboxImage.src = preloadedImages[imageKey].src;
                    lightboxImage.classList.add('loaded');
                    lightboxSpinner.style.display = 'none';
                } else {
                    lightboxImage.src = imgPath;
                    lightboxImage.onload = function() {
                        lightboxImage.classList.add('loaded');
                        lightboxSpinner.style.display = 'none';
                        preloadedImages[imageKey] = this;
                    };
                }
                
                // Preload następnych i poprzednich obrazków
                preloadImage(index - 1);
                preloadImage(index + 1);
            }

            // Zamykanie lightbox
            function closeLightboxFunc() {
                lightbox.style.display = 'none';
                document.body.style.overflow = 'auto';
            }

            // Nawigacja między zdjęciami
            function navigate(direction) {
                let newIndex = currentImageIndex;
                
                if (direction === 'prev' && currentImageIndex > 0) {
                    newIndex = currentImageIndex - 1;
                } else if (direction === 'next' && currentImageIndex < allImages.length - 1) {
                    newIndex = currentImageIndex + 1;
                } else {
                    return; // Nie ma gdzie nawigować
                }
                
                currentImageIndex = newIndex;
                
                // Pokaż spinner
                lightboxSpinner.style.display = 'block';
                lightboxImage.classList.remove('loaded');
                
                // Ustaw źródło obrazka
                const imgPath = `<?php echo $rootDir; ?>/${currentFolder}/${allImages[newIndex]}`;
                
                // Sprawdź, czy obrazek jest już załadowany
                const imageKey = `${currentFolder}_${newIndex}`;
                if (preloadedImages[imageKey]) {
                    lightboxImage.src = preloadedImages[imageKey].src;
                    lightboxImage.classList.add('loaded');
                    lightboxSpinner.style.display = 'none';
                } else {
                    lightboxImage.src = imgPath;
                    lightboxImage.onload = function() {
                        lightboxImage.classList.add('loaded');
                        lightboxSpinner.style.display = 'none';
                        preloadedImages[imageKey] = this;
                    };
                }
                
                // Preload następnych i poprzednich obrazków
                preloadImage(newIndex - 1);
                preloadImage(newIndex + 1);
            }

            // Aktualizacja URL galerii
            function updateGalleryUrl(folderPath) {
                const url = `${window.location.origin}${window.location.pathname}?folder=${encodeURIComponent(folderPath)}`;
                galleryUrl.textContent = url;
                history.pushState(null, null, `?folder=${encodeURIComponent(folderPath)}`);
            }

            // Kopiowanie URL
            copyBtn.addEventListener('click', function() {
                const url = galleryUrl.textContent;

                // Tworzymy tymczasowy element textarea
                const textarea = document.createElement('textarea');
                textarea.value = url;
                textarea.style.position = 'fixed';
                textarea.style.left = '-9999px';
                textarea.style.top = '-9999px';
                document.body.appendChild(textarea);
                textarea.select();

                try {
                    // Próbujemy skopiować
                    const successful = document.execCommand('copy');
                    if (successful) {
                        // Pokazujemy komunikat o sukcesie
                        copySuccess.style.display = 'block';
                        setTimeout(() => {
                            copySuccess.style.display = 'none';
                        }, 2000);
                    } else {
                        console.error('Nie udało się skopiować');
                    }
                } catch (err) {
                    console.error('Błąd podczas kopiowania:', err);
                }

                // Usuwamy textarea
                document.body.removeChild(textarea);
            });

            // Obsługa klawiszy
            document.addEventListener('keydown', function(e) {
                if (lightbox.style.display === 'flex') {
                    if (e.key === 'Escape') {
                        closeLightboxFunc();
                    } else if (e.key === 'ArrowLeft') {
                        navigate('prev');
                    } else if (e.key === 'ArrowRight') {
                        navigate('next');
                    }
                }
            });

            // Obsługa dotyku dla Lightbox
            lightbox.addEventListener('touchstart', function(e) {
                touchStartX = e.changedTouches[0].screenX;
            }, {passive: true});

            lightbox.addEventListener('touchend', function(e) {
                touchEndX = e.changedTouches[0].screenX;
                handleSwipe();
            }, {passive: true});

            function handleSwipe() {
                const threshold = 50; // minimalna odległość przesunięcia
                const diff = touchStartX - touchEndX;

                if (diff > threshold) {
                    // Przesunięcie w lewo - następne zdjęcie
                    navigate('next');
                } else if (diff < -threshold) {
                    // Przesunięcie w prawo - poprzednie zdjęcie
                    navigate('prev');
                }
            }

            // Event listeners
            closeLightbox.addEventListener('click', closeLightboxFunc);
            prevPhoto.addEventListener('click', () => navigate('prev'));
            nextPhoto.addEventListener('click', () => navigate('next'));

            // Zmienna PHP przekazana do JavaScript
            const initialFolder = '<?php echo $initialFolder; ?>';

            // Inicjalizacja
            loadFolderStructure('', explorer, true);
        });
    </script>
</body>
</html>